package de.ecconia.java.pnet.connector;

import java.io.IOException;
import java.net.BindException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.concurrent.locks.ReentrantLock;

import de.ecconia.java.pnet.exceptions.CloseException;
import de.ecconia.java.pnet.exceptions.CloseException.CloseInterruptedException;
import de.ecconia.java.pnet.exceptions.ServerException;
import de.ecconia.java.pnet.exceptions.ServerException.ClientCreationException;
import de.ecconia.java.pnet.exceptions.ServerException.ServerCrashException;
import de.ecconia.java.pnet.exceptions.StartException;
import de.ecconia.java.pnet.exceptions.StartException.UnexpectedStartException;

public class GenericServer
{
	private static int serverID = 0;
	
	private final int port;
	private SocketHandler socketHandler;
	private ServerRuntimeException exceptionHandler;
	
	private Thread serverThread;
	private ServerSocket server;
	
	//Lock to prevent .start() and .stop() to run at the same time.
	private ReentrantLock ssLock = new ReentrantLock();
	
	public GenericServer(int port)
	{
		this.port = port;
	}
	
	public void setSocketHandler(SocketHandler socketHandler)
	{
		ssLock.lock();
		
		if(serverThread != null)
		{
			ssLock.unlock();
			//TODO: This could be changed, by locking before grabbing a copy of SocketHandler on new Socket
			throw new IllegalStateException("Cannot change the SocketHandler, while the server is running.");
		}
		
		this.socketHandler = socketHandler;
		
		ssLock.unlock();
	}
	
	public void setServerRuntimeExceptionHandler(ServerRuntimeException exceptionHandler)
	{
		this.exceptionHandler = exceptionHandler;
	}
	
	public void start()
	{
		ssLock.lock();
		
		if(socketHandler == null)
		{
			ssLock.unlock();
			throw new IllegalStateException("Connection processor has not been set yet.");
		}
		
		if(serverThread != null)
		{
			ssLock.unlock();
			throw new IllegalStateException("Server is already running.");
		}
		
		try
		{
			server = new ServerSocket(port);
		}
		catch(IOException e)
		{
			if(e instanceof BindException)
			{
				if(e.getMessage().equals("Permission denied (Bind failed)"))
				{
					ssLock.unlock();
					throw new StartException("Cannot bind server on this port: No permission.");
				}
				else if(e.getMessage().equals("Address already in use (Bind failed)"))
				{
					ssLock.unlock();
					throw new StartException("Cannot bind server on this port: Port in use.");
				}
			}
			
			ssLock.unlock();
			throw new UnexpectedStartException(e);
		}
		
		serverThread = new Thread(() -> {
			try
			{
				while(true)
				{
					try
					{
						Socket socket = server.accept();
						
						new Thread(() -> {
							try
							{
								socketHandler.newSocket(socket);
							}
							catch(Exception e) //Since non-framework exceptions reach here, catch everything.
							{
								if(exceptionHandler != null)
								{
									exceptionHandler.newException(new ClientCreationException(e));
								}
							}
						}).start();
					}
					catch(SocketException e)
					{
						if("Socket closed".equals(e.getMessage()))
						{
							//Intended exception, occurs if close() has been called.
							return;
						}
						else
						{
							throw e;
						}
					}
				}
			}
			catch(Exception e)
			{
				if(exceptionHandler != null)
				{
					exceptionHandler.newException(new ServerCrashException(e));
				}
			}
		}, "ServerThread#" + (++serverID));
		serverThread.start();
		
		ssLock.unlock();
	}
	
	public void stop() throws CloseException
	{
		ssLock.lock();
		
		if(serverThread == null)
		{
			ssLock.unlock();
			throw new IllegalStateException("Server is not running.");
		}
		
		try
		{
			//Close everything related to the socket - and cause internal exceptions.
			if(server != null)
			{
				server.close();
				//Got closed just now.
				server = null;
			}
		}
		catch(IOException e)
		{
			//TBI: Which exception may happen here?
			throw new CloseException(e);
		}
		
		try
		{
			//Wait for this thread to really shutdown. Should be fast...
			serverThread.join();
			//Got destroyed just now.
			serverThread = null;
		}
		catch(InterruptedException e)
		{
			//If someone interrupts this thread, he must have a very good reason to.
			throw new CloseInterruptedException();
		}
		
		ssLock.unlock();
	}
	
	//TBI: Lock here too?
	public boolean isRunning()
	{
		return serverThread != null;
	}
	
	public interface SocketHandler
	{
		void newSocket(Socket socket);
	}
	
	public interface ServerRuntimeException
	{
		void newException(ServerException e);
	}

	public int getPort()
	{
		return port;
	}
}
